// SPDX-License-Identifier: BSD-3-Clause
/* Copyright(c) 2023 Broadcom
 * All rights reserved.
 */

#include <linux/types.h>
#include "bnxt_compat.h"
#include "tfo.h"
#include "cfa_types.h"
#include "cfa_tim.h"
#include "bnxt.h"

/* Table scope stored configuration */
struct tfc_tsid_db {
	bool ts_valid;			/* Table scope is valid */
	enum cfa_scope_type scope_type;	/* non-shared, shared-app, global */
	bool ts_is_bs_owner;		/* Backing store alloced by this instance (PF) */
	u16 ts_max_pools;		/* maximum pools per CPM instance */
	enum cfa_app_type ts_app;	/* application type TF/AFM */
	struct tfc_ts_mem_cfg ts_mem[CFA_REGION_TYPE_MAX][CFA_DIR_MAX]; /* backing store mem cfg */
	struct tfc_ts_pool_info ts_pool[CFA_DIR_MAX];			/* pool info config */
};

/* Only a single global scope is allowed
 */
#define TFC_GLOBAL_SCOPE_MAX 1

/* TFC Global Object
 * The global object is not per port, it is global.  It is only
 * used when a global table scope is created.
 */
struct tfc_global_object {
	u8 gtsid;					/* global tsid */
	struct tfc_tsid_db gtsid_db;			/* global db */
	void *gts_tim;					/* ts instance */
};

struct tfc_global_object tfc_global;

/* TFC Object Signature
 * This signature identifies the tfc object database and
 * is used for pointer validation
 */
#define TFC_OBJ_SIGNATURE 0xABACABAF

/* TFC Object
 * This data structure contains all data stored per bnxt port
 * Access is restricted through set/get APIs.
 *
 * If a module (e.g. tbl_scope needs to store data, it should
 * be added here and accessor functions created.
 */
struct tfc_object {
	u32 signature;					/* TF object signature */
	u16 sid;					/* Session ID */
	bool is_pf;					/* port is a PF */
	struct cfa_bld_mpcinfo mpc_info;		/* MPC ops handle */
	struct tfc_tsid_db tsid_db[TFC_TBL_SCOPE_MAX];	/* tsid database */
	/* TIM instance pointer (PF) - this is where the 4 instances
	 *  of the TPM (rx/tx_lkup, rx/tx_act) will be stored per
	 *  table scope.  Only valid on a PF.
	 */
	void *ts_tim;
	struct tfc_global_object *tfgo;			/* pointer to global */
};

/* Get the correct tsid database given the table scope id */
static inline struct tfc_tsid_db *get_tsid_db(struct tfc_object *tfco, u8 ts_tsid)
{
	struct tfc_global_object *tfgo;

	tfgo = tfco->tfgo;
	if (tfgo && tfgo->gtsid == ts_tsid)
		return &tfgo->gtsid_db;
	else
		return &tfco->tsid_db[ts_tsid];
}

void tfo_open(void **tfo, bool is_pf)
{
	struct tfc_object *tfco = NULL;
	struct tfc_global_object *tfgo;
	u32 tim_db_size;
	int rc;

	tfco = kzalloc(sizeof(*tfco), GFP_KERNEL);
	if (!tfco)
		return;

	tfco->signature = TFC_OBJ_SIGNATURE;
	tfco->is_pf = is_pf;
	tfco->sid = INVALID_SID;
	tfco->ts_tim = NULL;

	/* Bind to the MPC builder */
	rc = cfa_bld_mpc_bind(CFA_P70, &tfco->mpc_info);
	if (rc) {
		netdev_dbg(NULL, "%s: MPC bind failed\n", __func__);
		goto cleanup;
	}
	if (is_pf) {
		/* Allocate per bp TIM database */
		rc = cfa_tim_query(TFC_TBL_SCOPE_MAX, CFA_REGION_TYPE_MAX,
				   &tim_db_size);
		if (rc)
			goto cleanup;

		tfco->ts_tim = kzalloc(tim_db_size, GFP_KERNEL);
		if (!tfco->ts_tim)
			goto cleanup;

		rc = cfa_tim_open(tfco->ts_tim,
				  tim_db_size,
				  TFC_TBL_SCOPE_MAX,
				  CFA_REGION_TYPE_MAX);
		if (rc) {
			tfco->ts_tim = NULL;
			goto cleanup;
		}
	}
	/* Initialize global object
	 */
	tfco->tfgo = &tfc_global;
	tfgo = tfco->tfgo;

	if (is_pf && !tfgo->gts_tim) {
		/* Allocate global scope TIM database */
		rc = cfa_tim_query(TFC_GLOBAL_SCOPE_MAX + 1, CFA_REGION_TYPE_MAX,
				   &tim_db_size);
		if (rc)
			goto cleanup;

		tfgo->gts_tim = kzalloc(tim_db_size, GFP_KERNEL);
		if (!tfgo->gts_tim)
			goto cleanup;

		rc = cfa_tim_open(tfgo->gts_tim,
				  tim_db_size,
				  TFC_GLOBAL_SCOPE_MAX + 1,
				  CFA_REGION_TYPE_MAX);
		if (rc) {
			tfgo->gts_tim = NULL;
			goto cleanup;
		}
	}
	tfgo->gtsid = INVALID_TSID;
	*tfo = tfco;
	return;

cleanup:
	kfree(tfgo->gts_tim);
	kfree(tfco->ts_tim);
	kfree(tfco);
	*tfo = NULL;
}

void tfo_close(void **tfo)
{
	struct tfc_object *tfco = (struct tfc_object *)(*tfo);
	void *tim = NULL, *tpm = NULL;
	enum cfa_region_type region;
	int dir, rc, tsid;

	if (*tfo && tfco->signature == TFC_OBJ_SIGNATURE) {

		/* This loop will clean up both the global and the
		 * tfo table scope instances as tfo_tim_get() will
		 * get the appropriate tim instance.
		 */
		for (tsid = 0; tsid < TFC_TBL_SCOPE_MAX; tsid++) {
			if (tfo_tim_get(*tfo, &tim, tsid))
				continue;
			if (!tim)
				continue;
			for (region = 0; region < CFA_REGION_TYPE_MAX; region++) {
				for (dir = 0; dir < CFA_DIR_MAX; dir++) {
					tpm = NULL;
					rc = cfa_tim_tpm_inst_get(tim, tsid, region, dir, &tpm);
					if (!rc && tpm) {
						kfree(tpm);
						cfa_tim_tpm_inst_set(tim, tsid, region, dir, NULL);
					}
				}
			}
		}
		kfree(tfco->ts_tim);
		tfco->ts_tim = NULL;
		tfco->tfgo = NULL;
		kfree(*tfo);
		*tfo = NULL;
	}
}

int tfo_mpcinfo_get(void *tfo, struct cfa_bld_mpcinfo **mpc_info)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;

	if (!tfo)
		return -EINVAL;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	*mpc_info = &tfco->mpc_info;

	return 0;
}

int tfo_ts_validate(void *tfo, u8 ts_tsid, bool *ts_valid)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);
	if (ts_valid)
		*ts_valid = tsid_db->ts_valid;

	return 0;
}

int tfo_ts_set(void *tfo, u8 ts_tsid, enum cfa_scope_type scope_type,
	       enum cfa_app_type ts_app, bool ts_valid, u16 ts_max_pools)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_global_object *tfgo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	/* Set the correct tsid database based upon whether
	 * the scope type passed in indicates global
	 */
	tfgo = tfco->tfgo;
	if (scope_type == CFA_SCOPE_TYPE_GLOBAL) {
		tsid_db = &tfgo->gtsid_db;
		tfgo->gtsid = ts_tsid;
	} else if (scope_type == CFA_SCOPE_TYPE_INVALID && tfgo &&
		   ts_tsid == tfgo->gtsid) {
		tfgo->gtsid = INVALID_TSID;
		tsid_db = &tfgo->gtsid_db;
	} else {
		tsid_db = &tfco->tsid_db[ts_tsid];
	}

	tsid_db->ts_valid = ts_valid;
	tsid_db->scope_type = scope_type;
	tsid_db->ts_app = ts_app;
	tsid_db->ts_max_pools = ts_max_pools;

	return 0;
}

int tfo_ts_get(void *tfo, u8 ts_tsid, enum cfa_scope_type *scope_type,
	       enum cfa_app_type *ts_app, bool *ts_valid,
	       u16 *ts_max_pools)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);
	if (ts_valid)
		*ts_valid = tsid_db->ts_valid;

	if (scope_type)
		*scope_type = tsid_db->scope_type;

	if (ts_app)
		*ts_app = tsid_db->ts_app;

	if (ts_max_pools)
		*ts_max_pools = tsid_db->ts_max_pools;

	return 0;
}

/* Set the table scope memory configuration for this direction */
int tfo_ts_set_mem_cfg(void *tfo, uint8_t ts_tsid, enum cfa_dir dir,
		       enum cfa_region_type region, bool is_bs_owner,
		       struct tfc_ts_mem_cfg *mem_cfg)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!mem_cfg) {
		netdev_dbg(NULL, "%s: Invalid mem_cfg pointer\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);

	tsid_db->ts_mem[region][dir] = *mem_cfg;
	tsid_db->ts_is_bs_owner = is_bs_owner;

	return 0;
}

/* Get the table scope memory configuration for this direction */
int tfo_ts_get_mem_cfg(void *tfo, u8 ts_tsid, enum cfa_dir dir,
		       enum cfa_region_type region, bool *is_bs_owner,
		       struct tfc_ts_mem_cfg *mem_cfg)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!mem_cfg) {
		netdev_dbg(NULL, "%s: Invalid mem_cfg pointer\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);

	*mem_cfg = tsid_db->ts_mem[region][dir];
	if (is_bs_owner)
		*is_bs_owner = tsid_db->ts_is_bs_owner;

	return 0;
}

/* Get the Pool Manager instance */
int tfo_ts_get_cpm_inst(void *tfo, u8 ts_tsid, enum cfa_dir dir,
			struct tfc_cpm **cpm_lkup, struct tfc_cpm **cpm_act)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!cpm_lkup) {
		netdev_dbg(NULL, "%s: Invalid cpm_lkup pointer\n", __func__);
		return -EINVAL;
	}

	if (!cpm_act) {
		netdev_dbg(NULL, "%s: Invalid cpm_act pointer\n", __func__);
		return -EINVAL;
	}
	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}
	tsid_db = get_tsid_db(tfo, ts_tsid);

	*cpm_lkup = tsid_db->ts_pool[dir].lkup_cpm;
	*cpm_act = tsid_db->ts_pool[dir].act_cpm;

	return 0;
}

/* Set the Pool Manager instance */
int tfo_ts_set_cpm_inst(void *tfo, u8 ts_tsid, enum cfa_dir dir,
			struct tfc_cpm *cpm_lkup, struct tfc_cpm *cpm_act)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);

	tsid_db->ts_pool[dir].lkup_cpm = cpm_lkup;
	tsid_db->ts_pool[dir].act_cpm = cpm_act;

	return 0;
}

/* Set the table scope pool memory configuration for this direction */
int tfo_ts_set_pool_info(void *tfo, u8 ts_tsid, enum cfa_dir dir,
			 struct tfc_ts_pool_info *ts_pool)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!ts_pool) {
		netdev_dbg(NULL, "%s: Invalid ts_pool pointer\n", __func__);
		return -EINVAL;
	}

	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);
	tsid_db->ts_pool[dir] = *ts_pool;

	return 0;
}

/* Get the table scope pool memory configuration for this direction */
int tfo_ts_get_pool_info(void *tfo, u8 ts_tsid, enum cfa_dir dir,
			 struct tfc_ts_pool_info *ts_pool)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_tsid_db *tsid_db;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!ts_pool) {
		netdev_dbg(NULL, "%s: Invalid ts_pool pointer\n", __func__);
		return -EINVAL;
	}
	if (ts_tsid >= TFC_TBL_SCOPE_MAX) {
		netdev_dbg(NULL, "%s: Invalid tsid %d\n", __func__, ts_tsid);
		return -EINVAL;
	}

	tsid_db = get_tsid_db(tfo, ts_tsid);
	*ts_pool = tsid_db->ts_pool[dir];

	return 0;
}

int tfo_sid_set(void *tfo, uint16_t sid)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (tfco->sid != INVALID_SID && sid != INVALID_SID &&
	    tfco->sid != sid) {
		netdev_dbg(NULL, "%s: Cannot set SID %u, current session is %u.\n",
			   __func__, sid, tfco->sid);
		return -EINVAL;
	}

	tfco->sid = sid;

	return 0;
}

int tfo_sid_get(void *tfo, uint16_t *sid)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!sid) {
		netdev_dbg(NULL, "%s: Invalid sid pointer\n", __func__);
		return -EINVAL;
	}

	if (tfco->sid == INVALID_SID) {
		/* Session has not been created */
		return -ENODATA;
	}

	*sid = tfco->sid;

	return 0;
}

int tfo_tim_get(void *tfo, void **tim, u8 ts_tsid)
{
	struct tfc_object *tfco = (struct tfc_object *)tfo;
	struct tfc_global_object *tfgo;

	if (tfco->signature != TFC_OBJ_SIGNATURE) {
		netdev_dbg(NULL, "%s: Invalid tfo object\n", __func__);
		return -EINVAL;
	}

	if (!tim) {
		netdev_dbg(NULL, "%s: Invalid tim pointer to pointer\n", __func__);
		return -EINVAL;
	}

	/* Get the correct tsid database based upon whether
	 * the tsid passed in configured as the global tsid
	 */
	tfgo = tfco->tfgo;
	if (ts_tsid == tfgo->gtsid) {
		if (!tfgo->gts_tim)
		/* ts tim could be null, no need to log error message */
			return -ENODATA;
		*tim = tfgo->gts_tim;
	} else {
		if (!tfco->ts_tim)
			/* ts tim could be null, no need to log error message */
			return -ENODATA;
		*tim = tfco->ts_tim;
	}

	return 0;
}
