#!/bin/bash

DEVICE_NAME="$1"

ACTION="$2"

# Check if device exists
if [[ -z "$DEVICE_NAME" ]]; then
    exit 0
fi

sysfspath="/sys/class/infiniband/"
dpath="$sysfspath/$DEVICE_NAME/ports/1/gid_attrs/ndevs/0"
ipiface=`cat $dpath`

IS_LAG=0
# Logic to check for bond device
BOND_DEVICE="/sys/class/net/bonding_masters"
if [ -f "$BOND_DEVICE" ]; then
    bond=$(cat $BOND_DEVICE)
    if [[ "$ipiface" == "$bond" ]]; then
        intfs=$(cat /sys/class/net/$bond/bonding/slaves)
        read -r ipiface ipiface2 <<< "$intfs"
        IS_LAG=1
    fi
fi

#Check VF or PF
DEVICE_PATH="/sys/class/net/$ipiface/device"
VF_DEVICE_IDS=(
    "0x1606" "0x1607" "0x1608" "0x1609" "0x16bd" "0x16c1" "0x16c2"
    "0x16c3" "0x16c4" "0x16c5" "0x16cb" "0x16d3" "0x16dc" "0x16e1"
    "0x16e5" "0x16e6" "0x1806" "0x1807" "0x1808" "0x1809" "0x1819"
    "0xd800"
)

DEVICE_ID=$(cat "$DEVICE_PATH/device")
for ID in "${VF_DEVICE_IDS[@]}"; do
	if [ "$DEVICE_ID" == "$ID" ]; then
		exit 0
	fi
done

SETUP_LOG="/tmp/bnxt_setup_${DEVICE_NAME}.log"
MAX_LOG_SIZE=512000  # 500KB in bytes

# Check size of log file and rotate if necessary
if [ -f "$SETUP_LOG" ]; then
	log_size=$(stat -c%s "$SETUP_LOG")
        if [ "$log_size" -ge "$MAX_LOG_SIZE" ]; then
            rm "$SETUP_LOG"
            touch "$SETUP_LOG"
        fi
else
        touch "$SETUP_LOG"
fi

#Check if L2 interface after move action triggered in case NAME_PCI used
if [ "$ACTION" == "add" ] && [ -z "$ipiface" ]; then
        echo "Waiting for L2 interface to be loaded." >> "$SETUP_LOG"
        exit 0
fi

echo "Using RDMA device $DEVICE_NAME for interface $ipiface" >> "$SETUP_LOG"

SPECIFIC_CONFIG_FILE="/etc/bnxt_re/bnxt_re_${DEVICE_NAME}.conf"
DEFAULT_CONFIG_FILE="/etc/bnxt_re/bnxt_re.conf"

# Check if the specific config file exists
if [ -f "$SPECIFIC_CONFIG_FILE" ]; then
    CONFIG_FILE="$SPECIFIC_CONFIG_FILE"
    echo "Using device specific config file: $CONFIG_FILE" >> "$SETUP_LOG"
else
    CONFIG_FILE="$DEFAULT_CONFIG_FILE"
    echo "Using default config file: $CONFIG_FILE" >> "$SETUP_LOG"
fi

# Define Default Values
DEFAULT_FC_MODE=3
DEFAULT_ROCE_PRI=3
DEFAULT_ROCE_DSCP=26
DEFAULT_CNP_PRI=7
DEFAULT_CNP_DSCP=48
DEFAULT_ROCE_BW=50
DEFAULT_UTILITY=""

declare -A param_flags=(
    ["FC_MODE"]="-m"
    ["ROCE_PRI"]="-r"
    ["ROCE_DSCP"]="-s"
    ["CNP_PRI"]="-c"
    ["CNP_DSCP"]="-p"
    ["ROCE_BW"]="-b"
)

# Initialize parameters with default values
if [ ! -f "$CONFIG_FILE" ]; then
   echo "No Config file found.Using Default values" >> "$SETUP_LOG"
   ROCE_DSCP=$DEFAULT_ROCE_DSCP
fi
FC_MODE=$DEFAULT_FC_MODE
ROCE_PRI=$DEFAULT_ROCE_PRI
CNP_PRI=$DEFAULT_CNP_PRI
CNP_DSCP=$DEFAULT_CNP_DSCP
ROCE_BW=$DEFAULT_ROCE_BW
UTILITY=$DEFAULT_UTILITY

# Read each line in the config file
if [ -f "$CONFIG_FILE" ]; then
   while IFS='=' read -r key value; do
     if [[ ${param_flags["$key"]} ]]; then
         eval "$key='$value'"
     fi
     if [[ "$key" == "ENABLE_FC" ]]; then
        ENABLE_FC="$value"
        if [[ "$ENABLE_FC" == "0" ]]; then
    	   echo "Flow control is disabled. Exiting without running the setup." >> "$SETUP_LOG"
           exit 0
	fi
     fi
     if [[ "$key" == "FC_MODE" ]]; then
        FC_MODE="$value"
     fi
     if [[ "$FC_MODE" != "1" && "$key" == "ROCE_DSCP" ]]; then
    	if [[ -z $value ]]; then
	    ROCE_DSCP="$DEFAULT_ROCE_DSCP"
	fi
     fi
     if [[ "$key" == "UTILITY" ]]; then
	if [[ "$value" == "4" ]]; then
		if command -v dcb >/dev/null 2>&1; then
			UTILITY="$value"
			UTIL_FLAG="-u $UTILITY"
			echo "Using dcb for QoS configuration" >> "$SETUP_LOG"
		elif command -v niccli >/dev/null 2>&1; then
			UTILITY="3"
			UTIL_FLAG="-u $UTILITY"
			echo "dcb is not installed. Using niccli for QoS configuration" >> "$SETUP_LOG"
		else
			echo "Neither dcb nor niccli utilities are installed" >> "$SETUP_LOG"
		fi
	elif [[ "$value" == "3" ]]; then
		if command -v niccli >/dev/null 2>&1; then
			UTILITY="$value"
			UTIL_FLAG="-u $UTILITY"
			echo "Using niccli for QoS configuration" >> "$SETUP_LOG"
		else
			echo "niccli is not installed" >> "$SETUP_LOG"
		fi
	else
		echo "Utility values other than 4 or 3 unsupported" >> "$SETUP_LOG"
		exit 0
	fi
     fi

   done < "$CONFIG_FILE"
fi

export RUN_SETUP_SCRIPT=true
PCI_ADDRESS=$(ethtool -i $ipiface | grep 'bus-info' | awk '{print $2}')
SHORT_PCI_ADDRESS=$(echo $PCI_ADDRESS | awk -F':' '{print $2 ":" $3}')
MODEL=$(lspci | grep "$SHORT_PCI_ADDRESS" | awk '{for(i=1;i<=NF;i++) if($i ~ /BCM57[0-9][0-9]/) print $i}')

# Check if the model is Thor or wh+
if [[ $MODEL =~ BCM574[0-9]{2} ]]; then
	echo "GenP4 detected: $MODEL" >> "$SETUP_LOG"
	CNP_SERV_TYPE=0
else
	echo "GenP5 or later detected: $MODEL" >> "$SETUP_LOG"
	CNP_SERV_TYPE=1
fi

# Initialize the command string with the script name
if [[ "$IS_LAG" == "1" ]]; then
    echo "$DEVICE_NAME is a LAG device, programming both $ipiface and $ipiface2" >> "$SETUP_LOG"
    CMD="bash /usr/bin/bnxt_setupcc.sh -d $DEVICE_NAME -i $ipiface -i $ipiface2 -C $CNP_SERV_TYPE"
else
    CMD="bash /usr/bin/bnxt_setupcc.sh -d $DEVICE_NAME -i $ipiface -C $CNP_SERV_TYPE"
fi

# Detect available utilities and set the utility option if absent from config file
if [[ -z "$UTILITY" ]]; then
	if command -v dcb >/dev/null 2>&1; then
		UTIL_FLAG="-u 4"  # Use dcb
	elif command -v niccli >/dev/null 2>&1; then
		UTIL_FLAG="-u 3"
	else
		echo "Neither dcb nor niccli utilities are installed" >> "$SETUP_LOG"
    		exit 0
	fi
fi

# Construct the CMD string
for param in "${!param_flags[@]}"; do
    if [[ "$FC_MODE" == "1" ]]; then
	if [[ "$param" != "ROCE_PRI" && "$param" != "ROCE_BW" && "$param" != "FC_MODE" && "$param" != "ROCE_DSCP" ]]; then
            	unset param_flags["$param"]
    	fi
    fi
    flag=${param_flags["$param"]}
    value=$(eval echo \$$param)
    if [[ $flag ]]; then
       if [[ $value ]]; then
		CMD+=" $flag $value"
       fi
    fi
done

# Append the utility flag last to ensure it overrides any 'u' value from the config file
CMD+=" $UTIL_FLAG"
# Execute the command
$CMD >> "$SETUP_LOG" 2>&1
